
/*:
 * @target MZ
 * @plugindesc Optimizes parallel event processes by only activating when the player is in proximity.
 * @help This plugin reduces lag by only running parallel event processes when the player is standing in front of the event.
 */

(() => {
    const PROXIMITY_DISTANCE = 1; // Distance in tiles to activate the event
    const CHECK_INTERVAL = 10; // Number of frames between checks

    let frameCounter = 0;
    const activeEventIds = new Set();

    // Function to get the coordinates in front of the player based on their direction
    function getFrontCoordinates(player) {
        let x = player.x;
        let y = player.y;
        switch (player.direction()) {
            case 2: // Down
                y += 1;
                break;
            case 4: // Left
                x -= 1;
                break;
            case 6: // Right
                x += 1;
                break;
            case 8: // Up
                y -= 1;
                break;
        }
        return { x, y };
    }

    // Function to check if the player is facing the event and within the proximity distance
    function isPlayerNearEvent(player, event) {
        const distance = Math.abs(player.x - event.x) + Math.abs(player.y - event.y);
        return distance <= PROXIMITY_DISTANCE;
    }

    // Function to check if the player is facing the event and one tile away
    function isPlayerFacingEvent(player, event) {
        const frontCoordinates = getFrontCoordinates(player);
        return frontCoordinates.x === event.x && frontCoordinates.y === event.y;
    }

    // Main logic for the parallel event
    function updateProximityEvents() {
        $gameMap.events().forEach(event => {
            if (isPlayerNearEvent($gamePlayer, event)) {
                activeEventIds.add(event.eventId());
            } else {
                activeEventIds.delete(event.eventId());
            }
        });
    }

    // Override Game_CharacterBase.prototype.updateParallel with proximity check
    const _Game_Event_updateParallel = Game_Event.prototype.updateParallel;
    Game_Event.prototype.updateParallel = function() {
        if (activeEventIds.has(this.eventId())) {
            _Game_Event_updateParallel.call(this);
        }
    };

    // Main update function to be run as a parallel process
    function update() {
        frameCounter++;
        if (frameCounter >= CHECK_INTERVAL) {
            frameCounter = 0;
            updateProximityEvents();
        }
    }

    // Run update() method on a parallel event basis
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function(sceneActive) {
        _Game_Map_update.call(this, sceneActive);
        update();
    };
})();
